function [P,name] = evaluatefeatures_msse2016(data_win, timestamps, fs)
%
% Evaluate features for activity recognition 
% 
% Input arguments are:
% - "data_win" it is a DxW matrix where W is the number of windows and D the 
%   number of samples per window. data_win is not simply the 3d raw data but
%   it is the norm of acceleration recordings:
%   data = sqrt(ax.^2+ay.^2+az.^2); 
%   Acceleration data must be expressed in 'g' (not as integer values, g = 9.81 m*s^-2).
%   Before evaluating features a lowpass filter may be applied (e.g butterworth LP 20Hz, order 4)
% - "timestamps" is the vector of timestamps of samples in each window
% - "fs" is the nominal sampling frequency
% 
% Output arguments are:
% - "P" which contains the features
% - "name" with an identifier for each feature.
% 
% [Andrea Mannini: a.mannini@sssup.it; Created: 2012. Last changes: Oct 2016]


[dim, W] = size(data_win);


%% features
fsI = 90; change_sampling = 0;
%f_psd = 0:fs/512:fs/2;
f_psd = 0.3:0.1:15;  
    
window = 12800
ti = 1/fsI:1/fsI:(window/10^3);    
    
means_m = zeros(1,W); stds_m = zeros(1,W);
vars_m = zeros(1,W); maxs_m = zeros(1,W); mins_m = zeros(1,W);
p1 = zeros(1,W); p2 = zeros(1,W); f1 = zeros(1,W); f2 = zeros(1,W); ft = zeros(1,W);
pT = zeros(1,W); p1_pT_ratio = zeros(1,W); p1B = zeros(1,W); f1B = zeros(1,W);
wf1 = zeros(1,W); wf2 = zeros(1,W);
Pxx = zeros(length(f_psd ),W);
LowP = zeros(1,W); HiP = zeros(1,W); Hz2P = zeros(1,W);
LowP2 = zeros(1,W); HiP2 = zeros(1,W); Hz2P2 = zeros(1,W);
onset1 = zeros(1,W); onset2 = zeros(1,W); onset3 = zeros(1,W); onset4 = zeros(1,W);

for ww = 1:W
    
    dati2 = (data_win(:,ww));
    dati = dati2 - mean(dati2);
    
    if length(data_win(:,ww))<2
        dati = [];
    else
        % interp
        tm = ((timestamps - timestamps(1,1)) ./10^3) + 1/fs; % apr13
        % removing non unique time recordings.
        [t_w, ind] = unique(dati);
        ti = 1/fsI:1/fsI:(window/10^3);
        if(length(ind)>2)
            dati = interp1(tm(ind), dati(ind,:), ti);
        else
            dati = [];
        end
    end
    
    
    
    %% easy features
    means_m(:,ww) = squeeze(nanmean(dati2,1));
    stds_m(:,ww)  = squeeze(nanstd(dati2,1));
    vars_m(:,ww)  = squeeze(nanvar(dati2,1));
    
    if ((size(dati2,1)<2) || isempty(dati))
        maxs_m(:,ww)  = nan;
        mins_m(:,ww)  = nan;
        Pxx(:,ww) = nan.*(ones(size(f_psd)));
        p1(ww) = nan;
        p2(ww) = nan;
        f1(ww) = nan;
        f2(ww) = nan;
        ft(ww) = nan;
        pT(ww) = nan;
        p1_pT_ratio(ww) = nan;
        p1B(ww) = nan;
        f1B(ww) = nan;
        LowP(ww)  = nan; HiP(ww)  = nan; Hz2P(ww) = nan;
        LowP2(ww)  = nan; HiP2(ww)  = nan; Hz2P2(ww) = nan;
        onset1(ww)  = nan;onset2(ww)  = nan;onset3(ww)  = nan; onset4(ww)  = nan;
        
    elseif (means_m(:,ww)==0)&&(stds_m(:,ww)==0)
        disp(['WARNING: Window # ' num2str(ww) ' has only null data'])
        
        maxs_m(:,ww)  = 0;
        mins_m(:,ww)  = 0;
        Pxx(:,ww) = nan.*(ones(size(f_psd)));
        p1(ww) = nan;
        p2(ww) = nan;
        f1(ww) = nan;
        f2(ww) = nan;
        ft(ww) = nan;
        pT(ww) = nan;
        p1_pT_ratio(ww) = nan;
        p1B(ww) = nan;
        f1B(ww) = nan;
        LowP(ww)  = nan; HiP(ww)  = nan; Hz2P(ww) = nan;
        LowP2(ww)  = nan; HiP2(ww)  = nan; Hz2P2(ww) = nan;
        onset1(ww)  = nan;onset2(ww)  = nan;onset3(ww)  = nan; onset4(ww)  = nan;
        
    else
        
        maxs_m(:,ww)  = max(dati2);
        mins_m(:,ww)  = min(dati2);
       
        
        %% Frequency domain features (Zhang et al.)
%         [S,F,T,Pxx(:,ww)] = spectrogram(dati,length(dati),0,512,fs);
        [S,F,T,Pxx(:,ww)] = spectrogram(dati,length(dati),0,f_psd,fsI);
        
        
        [pks,locs] = findpeaks(Pxx(:,ww),'SORTSTR','descend');
        ind_allowed = find (F>0.3,1,'first');
        pks(locs<ind_allowed)= [];
        locs(locs<ind_allowed)= [];
        
        if (std(dati)<10^-10) %% sensor is not moving... 
            p1(ww) = 0;
            f1(ww) = eps;
            p2(ww) = 0;
            f2(ww) = eps;
            
            pT(ww) = eps;
        else
            
            p1(ww) = pks(1);
            f1(ww) = F(locs(1));
            if numel(pks)>1
                p2(ww) = pks(2);
                f2(ww) = F(locs(2));
            else
                p2(ww) = 0;
                f2(ww) = eps;
            end
            pT(ww) = sum(Pxx(:,ww));
            
        end
        
        p1_pT_ratio(ww) = p1(ww)/pT(ww);
            
        condit = F(locs)>=0.6 & F(locs)<=2.6;
        pks2 = pks(condit);
        locs2 = locs(condit);;
        
        try
            f1B(ww) = F(locs2(1));
            p1B(ww) = pks2(1);
        catch
            condit2 = ( F>=0.6 & F<=2.6);
            [p1B(ww) indF ] = max(Pxx(condit2,ww));
            ff = F(condit2);
            f1B(ww) = ff(indF);
        end
        
        if  ( ww>1 && ~isnan(f1(ww-1)))
            ft(ww) = f1(ww)/f1(ww-1);
        else
            ft(ww) = eps;
        end
        
        
        %% wavelet features
        Nwlt = 8;
        dati(isnan(dati))=[];
        [C,L] = wavedec(dati,Nwlt,'db10');
        
        clear det appr
        for i = 1:Nwlt
            det(i,:) = wrcoef('d',C,L,'db10',i);
            det_2(i) = det(i,:)*det(i,:)';
            % %             subplot(10,1,Nwlt-i+3), plot(det(i,:),'.-'), axis tight;
        end
        appr = wrcoef('a',C,L,'db10',Nwlt);
        
        det56_2 = det(5,:)* det(5,:)' + det(6,:)* det(6,:)';
        
        if (std(dati)<10^-10) %% sensor is not moving... 
            wf1(ww) = 0;
            wf2(ww) = 0;
        else
            wf1(ww) = det56_2 / ( (dati)*(dati)' );
            wf2(ww) = det56_2 /sum(det_2);
        end
        
        
        %% MSSE2016 features
        % % new features (sensor position recognition)
        condit1 = (F>=3);
        condit2 = (F<3);
        condit3 = (F>=1.5)&(F<=2.5);
        HiP(ww) = sum(Pxx(condit1,ww)) / pT(ww);
        LowP(ww)  = sum(Pxx(condit2,ww)) / pT(ww);
        Hz2P(ww) = sum(Pxx(condit3,ww)) / pT(ww);
       
        condit4 = (F>=3.5); %dataset "D" dec2013 (MSSE KIDS)
        condit5 = (F<3.5);
        
        condit6 = (F>=0.5)&(F<=1.5);
        HiP2(ww) = sum(Pxx(condit4,ww)) / pT(ww);
        LowP2(ww)  = sum(Pxx(condit5,ww)) / pT(ww);
        Hz2P2(ww) = sum(Pxx(condit6,ww)) / pT(ww);
        
        % % onset features (Fragmentation features )
        data_on = data_win;
        data_on = abs(data_on-1);
        [Bon,Aon] = butter(4, 5/fs,'low');
        data_on  = filtfilt(Bon,Aon,data_on);
        th_on = 0.2;
        onset1(ww) = sum(data_on>th_on) / length(data_on);
        if (sum(data_on>th_on))
            
            d_on = diff([0; data_on>th_on; 0]);
            onset2(ww) = sum(d_on==1) / sum(data_on>th_on);
            
            i_p1 = find(d_on == 1);
            i_m1 = find(d_on ==-1);
            dur_on = i_m1 - i_p1;
            onset3(ww) = mean(dur_on)/length(data_on);
            
            if length(dur_on)>1
                onset4(ww) = std(dur_on)/length(data_on);
            else
                onset4(ww) = 0;
            end
        else
            onset2(ww) = 0; onset3(ww) = 0; onset4(ww) = 0;
        end
            
        
    end
end

% P = [means_m; stds_m; vars_m; maxs_m; mins_m; maxs_m-mins_m; p1; p2; f1; f2; pT; p1_pT_ratio; p1B; f1B; ft; wf1; wf2;];
P = [means_m; stds_m; vars_m; maxs_m; mins_m; maxs_m-mins_m; p1; p2; f1; f2; pT; p1_pT_ratio; p1B; f1B; ft; wf1; wf2; HiP2; LowP2; Hz2P2; onset1; onset2; onset3; onset4;];
name = {'mean'; 'std'; 'var'; 'max'; 'min'; 'range'; 'p1'; 'p2'; 'f1'; 'f2'; 'pT'; 'p1_pT_ratio'; 'p1B'; 'f1B'; 'f_history'; 'waveletf1'; 'waveletf2'; 'HiP2'; 'LowP2'; 'Hz2P2'; 'F_S'; 'F_AN'; 'F_AM'; 'F_AV';};

